<?php

namespace Ademti\WoocommerceProductFeeds\Integrations;

use function is_callable;

/**
 * Integration for:
 * https://woocommerce.com/products/product-bundles/
 */
class ProductBundles {
	/**
	 * Add filters.
	 */
	public function run(): void {
		add_filter( 'woocommerce_gpf_wc_get_products_args', [ $this, 'add_products_to_query' ], 10, 2 );
		add_filter( 'woocommerce_gpf_product_price_calculator_callback', [ $this, 'assign_price_calculator' ], 10, 3 );
	}

	/**
	 * Add "bundle" to the list of queried-for product types when relevant.
	 *
	 * @param $args
	 * @param $type
	 *
	 * @return mixed
	 */
	public function add_products_to_query( $args, $type ) {
		if ( in_array( $type, [ 'feed', 'status', 'RebuildSimpleJob' ], true ) ) {
			$args['type'][] = 'bundle';
		}
		return $args;
	}

	/**
	 * Register our custom price calculator method for composite products.
	 *
	 * @param $calculator
	 * @param $product_type
	 * @param $product
	 *
	 * @return array
	 * @SuppressWarnings(PHPMD.UnusedFormalParameter)
	 */
	// phpcs:disable Generic.CodeAnalysis.UnusedFunctionParameter.FoundAfterLastUsed
	public function assign_price_calculator( $calculator, $product_type, $product ) {
		if ( 'bundle' === $product_type ) {
			return [ $this, 'calculate_prices' ];
		}
		return $calculator;
	}
	// phpcs:enable Generic.CodeAnalysis.UnusedFunctionParameter.FoundAfterLastUsed

	/**
	 * Calculate prices for Bundle products.
	 * @param $product
	 * @param $prices
	 *
	 * @return array
	 */
	public function calculate_prices( $product, $prices ) {
		// Use tax-specific functions if available.
		if ( is_callable( [ $product, 'get_bundle_regular_price_including_tax' ] ) ) {
			$prices['regular_price_ex_tax']  = $product->get_bundle_regular_price_excluding_tax();
			$prices['regular_price_inc_tax'] = $product->get_bundle_regular_price_including_tax();
			$current_price_ex_tax            = $product->get_bundle_price_excluding_tax();
			if ( $current_price_ex_tax < $prices['regular_price_ex_tax'] ) {
				$prices['sale_price_ex_tax']  = $product->get_bundle_price_excluding_tax();
				$prices['sale_price_inc_tax'] = $product->get_bundle_price_including_tax();
			}
		} else {
			// Just take the current price as the regular price since its
			// the only one we can reliably get.
			$prices['regular_price_ex_tax']  = $product->get_bundle_price_excluding_tax();
			$prices['regular_price_inc_tax'] = $product->get_bundle_price_including_tax();
		}
		// Populate a "price", using the sale price if there is one, the actual price if not.
		if ( '' !== $prices['sale_price_ex_tax'] ) {
			$prices['price_ex_tax']  = $prices['sale_price_ex_tax'];
			$prices['price_inc_tax'] = $prices['sale_price_inc_tax'];
		} else {
			$prices['price_ex_tax']  = $prices['regular_price_ex_tax'];
			$prices['price_inc_tax'] = $prices['regular_price_inc_tax'];
		}

		return $prices;
	}
}
